﻿using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.Threading.Tasks;
using System.Net.Http;
using System.Net.Http.Headers;
using System.IO;
using System.Text;
using System.Threading;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using WebWindow;

namespace OneDrive
{
    public partial class OneDriveControl
    {
        private async Task<MemoryStream> DownloadPart(OneDrive.ResponseFolder target, int index = 0, CancellationToken ct = default(CancellationToken))
        {
            MemoryStream ret = new MemoryStream();
            int rangemax = concurrency;
            if (target.size < (index + 1) * concurrency * DownloadBlock)
            {
                rangemax = (Int32)((target.size-1) / DownloadBlock + 1) % concurrency;
            }
            ct.ThrowIfCancellationRequested();
            // Create a query. 
            IEnumerable<Task<MemoryStream>> downloadTasksQuery =
                from x in Enumerable.Range(0, rangemax) select DownloadItem(target, x + index * concurrency, ct);

            // Use ToArray to execute the query and start the download tasks.
            Task<MemoryStream>[] downloadTasks = downloadTasksQuery.ToArray();

            // Await the completion of all the running tasks.
            MemoryStream[] buf = await Task.WhenAll(downloadTasks);
            ct.ThrowIfCancellationRequested();
            foreach (var abuf in buf)
            {
                abuf.WriteTo(ret);
                abuf.Dispose();
            }
            ret.Position = 0;
            return ret;
        }

        private async Task SendUDP(OneDrive.ResponseFolder target, double startsec = 0, double durationsec = 0, CancellationToken cancellationToken = default(CancellationToken))
        {
            if (target.size == 0) return;
            var splitcount = (Int32)((target.size - 1) / (concurrency * DownloadBlock)) + 1;
            int count = 0;
            int bufcount = -1;
            Int64 FirstSkip = 0;
            MemoryStream buf = null;
            Queue<Task<MemoryStream>> downloadTask = new Queue<Task<MemoryStream>>();
            Queue<MemoryStream> bufQueue = new Queue<MemoryStream>();

            var sender = new StreamSender(cancellationToken);
            sender.SendDuration = TimeSpan.FromSeconds(durationsec);

            if (startsec > 0)
            {
                Int64 bytePerSec = 0;
                buf = await DownloadPart(target, count, ct: cancellationToken);
                bufcount = 0;
                bytePerSec = await sender.Calc1secBytes(buf);
                while (bytePerSec == 0 && ++count < splitcount)
                {
                    buf = await DownloadPart(target, count, cancellationToken);
                    bufcount = count;
                    bytePerSec = await sender.Calc1secBytes(buf);
                }
                Int64 SkipBytes = (Int64)(bytePerSec * startsec);
                count = (int)(SkipBytes / (concurrency * DownloadBlock));
                FirstSkip = SkipBytes % (concurrency * DownloadBlock);
            }
            if (count == bufcount)
            {
                bufQueue.Enqueue(buf);
            }
            else
            {
                downloadTask.Enqueue(DownloadPart(target, count, ct: cancellationToken));
            }
            bool init = false;
            while (++count < splitcount)
            {
                if (downloadTask.Count > 1) bufQueue.Enqueue(await downloadTask.Dequeue());
                downloadTask.Enqueue(DownloadPart(target, count, ct: cancellationToken));
                if (bufQueue.Count > 2)
                {
                    buf = bufQueue.Dequeue();
                    if (!init)
                    {
                        buf.Position = FirstSkip;
                        sender.Init();
                        init = true;
                    }
                    if (buf != null) await sender.SendPacket(buf);
                }
            }
            while(downloadTask.Count > 0)
            {
                bufQueue.Enqueue(await downloadTask.Dequeue());
                await sender.SendPacket(bufQueue.Dequeue());
            }
            while(bufQueue.Count > 0)
            {
                await sender.SendPacket(bufQueue.Dequeue());
            }
        }

        private async Task SendUDPMultiBlock(string target_id, double startsec = 0, double durationsec = 0, CancellationToken cancellationToken = default(CancellationToken))
        {
            byte[] salt;
            string password = Config.password_prefix;
            string orgFilename;
            byte[] iv;
            var rijndael = new RijndaelManaged();
            rijndael.KeySize = 256;

            // follow path
            FolderRequest children = await GetFolder_id(target_id, cancellationToken);
            ResponseFolder mother = await GetFile_id(target_id, cancellationToken);

            int num = 0;
            var header = children.value.Find(x => x.name == "0000");
            if (header != null)
            {
                var head = await DownloadPart(header, 0, cancellationToken);
                byte[] buf = new byte[sizeof(Int32)];
                head.Read(buf, 0, sizeof(Int32));
                num = BitConverter.ToInt32(buf, 0);

                int ver = (num & 0x7F000000) >> 24;
                num &= 0xFFFFFF;

                if (ver == 0)
                    rijndael.BlockSize = 256;
                else
                    rijndael.BlockSize = 128;

                iv = new byte[rijndael.BlockSize / 8];
                head.Read(iv, 0, iv.Length);

                salt = new byte[SaltSize];
                head.Read(salt, 0, salt.Length);
                int strlen = 0;
                for (int i = 0; i < 4; i++)
                {
                    byte[] p = new byte[1];
                    head.Read(p, 0, 1);
                    strlen |= ((p[0] & 0x7F) << 7 * i);
                    if ((p[0] & 0x80) == 0) break;
                }
                var strbuf = new byte[strlen];
                head.Read(strbuf, 0, strbuf.Length);
                orgFilename = Encoding.Unicode.GetString(strbuf);
            }
            else return;
            if (num <= 0) return;
            if ((mother.folder == null) || (mother.folder.childCount == null)) return;
            if (mother.folder.childCount != num + 1) return;

            password += orgFilename;
            var DeriveBytes = new Rfc2898DeriveBytes(password, salt, myIterations);
            rijndael.Key = DeriveBytes.GetBytes(rijndael.KeySize / 8);
            rijndael.IV = iv;
            ICryptoTransform decryptor = rijndael.CreateDecryptor();
            DeriveBytes.Reset();

            int concurrency = 4;
            children.value.RemoveAll(x => x.name == "0000");

            var sender = new StreamSender(cancellationToken);
            sender.startsec = startsec;
            sender.SendDuration = TimeSpan.FromSeconds(durationsec);

            // to break internal
            var internalCanceler = new CancellationTokenSource();
            var linkedCts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, internalCanceler.Token);
            var ctintarnal = linkedCts.Token;

            Task DecryptTask = null;
            Queue<Task<StreamStock>[]> downloadTasks = new Queue<Task<StreamStock>[]>();
            Queue<StreamStock> bufQueue = new Queue<StreamStock>();
            Task SendTask = null;
            Int64 SkipBytes = (startsec > 0)? -1: 0;
            Int64 DownloadingHeadByte = 0;
            Int64 DecryptHeadOffset = 0;
            bool init = false;
            IDecryptStream dest = new DecryptStreamSender(decryptor, ctintarnal);
            foreach (var downtask in from aFile in children.value orderby int.Parse(aFile.name) group aFile by int.Parse(aFile.name) / concurrency)
            {
                try
                {
                    var downloadoffset = (from prevFile in children.value where int.Parse(prevFile.name) < int.Parse(downtask.First().name) select prevFile.size).Sum();
                    System.Diagnostics.Debug.WriteLine("loop start : offset " + downloadoffset.ToString());
                    DownloadingHeadByte = downloadoffset ?? 0;

                    if (downloadTasks.Count > 1)
                    {
                        // Await the completion of all the running tasks.
                        var allbuf = await Task.WhenAll(downloadTasks.Dequeue());
                        System.Diagnostics.Debug.WriteLine("Download done. : offset " + allbuf.First().Offset.ToString());

                        if (DecryptTask != null)
                        {
                            await DecryptTask;
                        }
                        DecryptTask = (Task.Run(async () =>
                        {
                            DecryptHeadOffset = allbuf.First().Offset;
                            MemoryStream ms = new MemoryStream();
                            foreach (var buf in allbuf)
                            {
                                ctintarnal.ThrowIfCancellationRequested();
                                await dest.DecryptStream(ms, buf.Stream);
                            }
                            ms.Position = 0;
                            System.Diagnostics.Debug.WriteLine("Decrypt done. : offset " + DecryptHeadOffset.ToString());
                            bufQueue.Enqueue(new StreamStock(ms, DecryptHeadOffset));
                        }, ctintarnal));
                    }
                    Int64 bytePerSec = 0;
                    var nextProcessing = (Int64)(from aFile in downtask select aFile.size).Sum();

                    // check download skip
                    if (SkipBytes >= 0 && DownloadingHeadByte + nextProcessing < SkipBytes)
                    {
                        System.Diagnostics.Debug.WriteLine("Download skip nowpos  " + DownloadingHeadByte.ToString());
                        System.Diagnostics.Debug.WriteLine("Download skip process " + nextProcessing.ToString());
                        DownloadingHeadByte = downloadoffset ?? 0 + nextProcessing;
                        continue;
                    }

                    System.Diagnostics.Debug.WriteLine("Download start:" + downloadoffset.ToString());
                    // Create a query. 
                    IEnumerable<Task<StreamStock>> downloadTasksQuery =
                        from aFile in downtask select (Task.Run(async () =>
                    {
                        var offset = (from prevFile in children.value where int.Parse(prevFile.name) < int.Parse(aFile.name) select prevFile.size).Sum();
                        return new StreamStock(await DownloadBuf(aFile, cancellationToken), offset ?? 0);
                    },cancellationToken));

                    // Use ToArray to execute the query and start the download tasks.
                    downloadTasks.Enqueue(downloadTasksQuery.ToArray());
                    DownloadingHeadByte = downloadoffset??0 + nextProcessing;

                    if (startsec > 0 && SkipBytes < 0)
                    {
                        if (bufQueue.Count > 0)
                        {
                            bytePerSec = await sender.Calc1secBytes(bufQueue.Peek().Stream);
                            if (bytePerSec == 0)
                            {
                                System.Diagnostics.Debug.WriteLine("cannot detect TOT in TS");
                                break;
                            }
                            SkipBytes = (Int64)(bytePerSec * startsec);
                            System.Diagnostics.Debug.WriteLine("Bps " + bytePerSec.ToString() + "; Skip " + SkipBytes.ToString());
                        }
                    }
                    if (bufQueue.Count > 0)
                    {
                        var buf = bufQueue.Dequeue();
                        var ms = buf.Stream;
                        if (!init)
                        {
                            if (SkipBytes - buf.Offset < buf.Stream.Length)
                            {
                                ms.Position = SkipBytes - buf.Offset;
                                System.Diagnostics.Debug.WriteLine("init stream: skip " + SkipBytes.ToString());
                                System.Diagnostics.Debug.WriteLine("init stream: buf len " + ms.Length.ToString());
                                System.Diagnostics.Debug.WriteLine("init stream: buf offset " + ms.Position.ToString());
                                sender.Init();
                                init = true;
                            }
                            else
                            {
                                System.Diagnostics.Debug.WriteLine("skip toOffset   :" + SkipBytes.ToString());
                                System.Diagnostics.Debug.WriteLine("skip Offset     :" + buf.Offset.ToString());
                                System.Diagnostics.Debug.WriteLine("skip Downloading:" + DownloadingHeadByte.ToString());
                                if (DownloadingHeadByte < SkipBytes)
                                {
                                    System.Diagnostics.Debug.WriteLine("cancel to download");
                                    internalCanceler.Cancel();
                                }
                                continue;
                            }
                        }
                        if (SendTask != null) await SendTask;
                        SendTask = sender.SendPacket(ms);
                    }
                }
                catch (OperationCanceledException)
                {
                    if (internalCanceler.Token.IsCancellationRequested)
                    {
                        internalCanceler = new CancellationTokenSource();
                        linkedCts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, internalCanceler.Token);
                        ctintarnal = linkedCts.Token;
                        dest.SetCancellationToken(ctintarnal);
                        sender.SetCancellationToken(ctintarnal);
                        SendTask = null;
                        DecryptTask = null;
                        downloadTasks.Clear();
                    }
                    else
                        throw;
                }
            }
            while (downloadTasks.Count > 0)
            {
                // Await the completion of all the running tasks.
                var allbuf = await Task.WhenAll(downloadTasks.Dequeue());
                System.Diagnostics.Debug.WriteLine("Download done. : offset " + allbuf.First().Offset.ToString());

                if (DecryptTask != null)
                {
                    await DecryptTask;
                }
                DecryptTask = (Task.Run(async () =>
                {
                    DecryptHeadOffset = allbuf.First().Offset;
                    MemoryStream ms = new MemoryStream();
                    foreach (var buf in allbuf)
                    {
                        ctintarnal.ThrowIfCancellationRequested();
                        await dest.DecryptStream(ms, buf.Stream);
                    }
                    ms.Position = 0;
                    System.Diagnostics.Debug.WriteLine("Decrypt done. : offset " + DecryptHeadOffset.ToString());
                    bufQueue.Enqueue(new StreamStock(ms, DecryptHeadOffset));
                }, ctintarnal));
            }
            if (DecryptTask != null)
            {
                await DecryptTask;
            }
            while (bufQueue.Count > 0)
            {
                var buf = bufQueue.Dequeue();
                if (!init)
                {
                    break;
                }
                if (buf != null)
                {
                    if (SendTask != null) await SendTask;
                    SendTask = sender.SendPacket(buf.Stream);
                }
            }
            if (SendTask != null) await SendTask;
            try
            {
                using (MemoryStream ms = new MemoryStream())
                {
                    await dest.DecryptStreamFinal(ms);
                    ms.Position = 0;
                    if (init) await sender.SendPacket(ms);
                }
            }
            catch (CryptographicException ex)
            {
                System.Diagnostics.Debug.WriteLine(ex.ToString());
            }
        }

        public async Task SendUDP_byPath(string target, double startsec = 0, double durationsec = 0, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            var target_folder = await GetFile_path(target, cancellationToken);
            await SendUDP_byFolder(target_folder, startsec, durationsec, cancellationToken);
        }

        public async Task SendUDP_byId(string target_id, double startsec = 0, double durationsec = 0, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            var target_folder = await GetFile_id(target_id, cancellationToken);
            await SendUDP_byFolder(target_folder, startsec, durationsec, cancellationToken);
        }

        private async Task SendUDP_byFolder(ResponseFolder target_folder, double startsec = 0, double durationsec = 0, CancellationToken cancellationToken = default(CancellationToken))
        {
            if (target_folder == null) return;

            if (target_folder.folder != null)
            {
                await SendUDPMultiBlock(target_folder.id, startsec, durationsec, cancellationToken);
                return;
            }

            await SendUDP(target_folder, startsec, durationsec, cancellationToken);
        }
    }

    public class StreamStock
    {
        public MemoryStream Stream;
        public Int64 Offset;
        public StreamStock(MemoryStream aStreem, Int64 aOffset)
        {
            Stream = aStreem;
            Offset = aOffset;
        }
    }
    public class StreamSender
    {
        const int packetlen = 188 * 50;
        byte[] packet = new byte[packetlen];
        DateTime StartTime;
        DateTime InitialTime;
        public TimeSpan SendDuration = TimeSpan.FromSeconds(0);
        bool TOTinit;
        int packet_last;
        Int64 bytecounter;
        CancellationToken ct;

        public StreamSender(CancellationToken cancellationToken = default(CancellationToken))
        {
            Init();
            ct = cancellationToken;
        }

        public void Init()
        {
            StartTime = DateTime.Now;
            TOTinit = false;
            packet_last = 0;
        }

        public void SetCancellationToken(CancellationToken cancellationToken)
        {
            this.ct = cancellationToken;
        }

        public async Task<Int64> Calc1secBytes(Stream data)
        {
            Int64 ret = 0;
            await Task.Run(() =>
            {
                var data_remain = data.Length;
                while (data.Position < data.Length)
                {
                    ct.ThrowIfCancellationRequested();
                    if (data_remain < packetlen)
                    {
                        data.Read(packet, 0, (int)data_remain);
                        packet_last = (int)data_remain;
                        break;
                    }
                    data.Read(packet, packet_last, packetlen - packet_last);
                    data_remain -= (packetlen - packet_last);
                    packet_last = 0;

                    GCHandle gch = GCHandle.Alloc(packet, GCHandleType.Pinned);
                    for (var j = 0; j < packetlen; j += 188)
                    {
                        var TOT = (TS_packet.TOT_transport_packet)Marshal.PtrToStructure(gch.AddrOfPinnedObject() + j, typeof(TS_packet.TOT_transport_packet));
                        while (!TOT.IsSync && ++j < packetlen)
                        {
                            TOT = (TS_packet.TOT_transport_packet)Marshal.PtrToStructure(gch.AddrOfPinnedObject() + j, typeof(TS_packet.TOT_transport_packet));
                        }
                        if (TOT.IsTOT)
                        {
                            if (!TOTinit)
                            {
                                InitialTime = TOT.JST_time;
                                TOTinit = true;
                                bytecounter = 0;
                            }
                            var sendspent = TOT.JST_time - InitialTime;

                            if (sendspent > TimeSpan.FromSeconds(5))
                            {
                                ret = (Int64)(bytecounter / sendspent.TotalSeconds);
                            }
                        }
                        bytecounter += 188;
                    }
                    gch.Free();
                }
            }, ct);
            return ret;
        }

        public async Task SendPacket(Stream data, bool dispose = false)
        {
            await Task.Run(() =>
            {
                using (var udp = new System.Net.Sockets.UdpClient())
                {
                    var data_remain = data.Length - data.Position;
                    while (data.Position < data.Length)
                    {
                        ct.ThrowIfCancellationRequested();
                        if (data_remain < packetlen)
                        {
                            data.Read(packet, 0, (int)data_remain);
                            packet_last = (int)data_remain;
                            break;
                        }
                        data.Read(packet, packet_last, packetlen - packet_last);
                        data_remain -= (packetlen - packet_last);
                        packet_last = 0;
                        udp.SendAsync(packet, packetlen, Config.SendToHost, Config.SentToPort);
                        System.Threading.Thread.Sleep(3);

                        GCHandle gch = GCHandle.Alloc(packet, GCHandleType.Pinned);
                        for (var j = 0; j < packetlen; j += 188)
                        {
                            var TOT = (TS_packet.TOT_transport_packet)Marshal.PtrToStructure(gch.AddrOfPinnedObject() + j, typeof(TS_packet.TOT_transport_packet));
                            while (!TOT.IsSync && ++j < packetlen)
                            {
                                TOT = (TS_packet.TOT_transport_packet)Marshal.PtrToStructure(gch.AddrOfPinnedObject() + j, typeof(TS_packet.TOT_transport_packet));
                            }
                            if (TOT.IsTOT)
                            {
                                if (!TOTinit)
                                {
                                    StartTime = DateTime.Now;
                                    InitialTime = TOT.JST_time;
                                    TOTinit = true;
                                }
                                var timespent = DateTime.Now - StartTime;
                                var sendspent = TOT.JST_time - InitialTime;
                                if ((SendDuration > TimeSpan.FromSeconds(0)) && (sendspent > SendDuration))
                                {
                                    throw new OperationCanceledException();
                                }

                                if (sendspent - timespent > TimeSpan.FromSeconds(5))
                                {
                                    System.Threading.Thread.Sleep(sendspent - timespent);
                                }
                            }
                        }
                        gch.Free();
                    }
                }
                if (dispose) data.Dispose();
            }, ct);
        }

        public Int64 SkipBytes = 0;
        public Int64 ProcessBytes = 0;
        public double startsec = 0;
        bool init = false;

        public async Task SendProcess(Stream destination)
        {
            await Task.Run(async () =>
            {
                if (startsec > 0 && SkipBytes == 0)
                {
                    Int64 bytePerSec = 0;
                    bytePerSec = await Calc1secBytes(destination);
                    SkipBytes = (Int64)(bytePerSec * startsec);
                    System.Diagnostics.Debug.WriteLine("Bps " + bytePerSec.ToString() + "; Skip " + SkipBytes.ToString());
                }
                if (SkipBytes - ProcessBytes > 0)
                {
                    if (SkipBytes - ProcessBytes > destination.Length)
                        ProcessBytes += destination.Length;
                    else
                    {
                        destination.Position = SkipBytes - ProcessBytes;
                        if (destination.Length - destination.Position > 32 * 1024 * 1024)
                        {
                            ProcessBytes += SkipBytes - ProcessBytes;
                        }
                        else
                        {
                            destination.Position = destination.Length;
                            return;
                        }
                    }
                }
                if (ProcessBytes >= SkipBytes)
                {
                    ProcessBytes += destination.Length - destination.Position;
                    if (!init)
                    {
                        Init();
                        init = true;
                    }
                    await SendPacket(destination);
                }
                if (!(startsec > 0 && SkipBytes == 0))
                    destination.Position = 0;
            }, ct);
        }

        public bool IsSkipping(Int64 nextProcessingBytes)
        {
            if (SkipBytes > 0)
            {
                if (ProcessBytes < SkipBytes)
                {
                    if (nextProcessingBytes + ProcessBytes < SkipBytes)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        public void Skip(Int64 skip)
        {
            ProcessBytes += skip;
        }

    }

    public class DecryptStreamSender: IDecryptStream
    {
        private byte[] InternalBuffer = null;
        ICryptoTransform decryptor;
        CancellationToken ct;

        public DecryptStreamSender(ICryptoTransform decryptor, CancellationToken cancellationToken = default(CancellationToken))
        {
            this.decryptor = decryptor;
            this.ct = cancellationToken;
        }

        public void SetCancellationToken(CancellationToken cancellationToken)
        {
            this.ct = cancellationToken;
        }

        public async Task DecryptStream(Stream outStream, Stream buf)
        {
            await Task.Run(() =>
            {
                byte[] outbuffer = new byte[decryptor.OutputBlockSize];
                buf.Position = 0;
                while (buf.Position < buf.Length)
                {
                    ct.ThrowIfCancellationRequested();
                    if (InternalBuffer != null)
                    {
                        var bytesWrite = decryptor.TransformBlock(InternalBuffer, 0, InternalBuffer.Length, outbuffer, 0);
                        outStream.Write(outbuffer, 0, bytesWrite);
                    }
                    if (buf.Length - buf.Position < decryptor.InputBlockSize) throw new DataMisalignedException();
                    InternalBuffer = new byte[decryptor.InputBlockSize];
                    buf.Read(InternalBuffer, 0, InternalBuffer.Length);
                }
            }, ct);
        }

        public async Task DecryptStreamFinal(Stream outStream)
        {
            await Task.Run(() =>
            {
                if (InternalBuffer != null)
                {
                    var outbuffer = decryptor.TransformFinalBlock(InternalBuffer, 0, InternalBuffer.Length);
                    outStream.Write(outbuffer, 0, outbuffer.Length);
                }
            }, ct);
        }
    }
}
