﻿using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.Threading.Tasks;
using System.Net.Http;
using System.Net.Http.Headers;
using System.IO;
using System.Text;
using System.Threading;
using System.Linq;
using System.Runtime.InteropServices;
using WebWindow;

namespace OneDrive
{
    [DataContract]
    public class ParentRef
    {
        [DataMember]
        public string driveId;
        [DataMember]
        public string id;
        [DataMember]
        public string path;
    }

    [DataContract]
    public class Hashes
    {
        [DataMember]
        public string sha1Hash;
        [DataMember]
        public string crc32Hash;
    }

    [DataContract]
    public class FileItem
    {
        [DataMember]
        public Hashes hashes;
        [DataMember]
        public string mimeType;
    }

    [DataContract]
    public class FolderItem
    {
        [DataMember]
        public Int64? childCount;
    }

    [DataContract]
    public class ResponseFolder
    {
        public bool loaded = false;
        [DataMember]
        public string createdDateTime;
        [DataMember]
        public string cTag;
        [DataMember]
        public string eTag;
        [DataMember]
        public string id;
        [DataMember]
        public string lastModifiedDateTime;
        [DataMember]
        public string name;
        [DataMember]
        public ParentRef parentReference;
        [DataMember]
        public Int64? size;
        [DataMember]
        public FolderItem folder;
        [DataMember]
        public FileItem file;
    }

    [DataContract]
    public class FolderRequest
    {
        [DataMember]
        public List<ResponseFolder> value;
        [DataMember(Name = "@odata.nextLink")]
        public string nextLink;
    }

    [DataContract]
    public class SearchRequest
    {
        [DataMember(Name = "@search.approximateCount")]
        public Int64? count;
        [DataMember]
        public List<ResponseFolder> value;
        [DataMember(Name = "@odata.nextLink")]
        public string nextLink;
    }

    [DataContract]
    public class UploadSession
    {
        [DataMember]
        public string uploadUrl;
        [DataMember]
        public string expirationDateTime;
        [DataMember]
        public List<string> nextExpectedRanges;
    }

    public class UploadResume
    {
        string access_token;
        public string errorstr { get; private set; }
        private Stream upcontent;
        private string filename;
        private string url;
        private string uploadurl;
        Int64 TotalByte;
        Int64 offset;

        public UploadResume(Stream uploadcontent, string filename, string url)
        {
            this.upcontent = uploadcontent;
            this.filename = filename;
            this.url = url;
        }
        public async Task<string> Upload()
        {
            string ret = "";
            var session = await CreateUploadSession();
            if(session == null) return null;
            uploadurl = session.uploadUrl;
            TotalByte = upcontent.Length;
            offset = 0;
            MemoryStream spstream;
            while ((spstream = SplitStreamNext()) != null)
            {
                ret = await UploadContent_Fragment(spstream);
            }
            return ret;
        }

        private MemoryStream SplitStreamNext()
        {
            if (upcontent.Position == upcontent.Length) return null;
            int splitsize = 50 * 1024 * 1024;
            int len = (int)Math.Min(splitsize, upcontent.Length - upcontent.Position);
            var buf = new byte[len];
            var ret = new MemoryStream();
            upcontent.Read(buf, 0, len);
            ret.Write(buf, 0, len);
            ret.Position = 0;
            return ret;
        }

        private async Task<UploadSession> CreateUploadSession(CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            using (var client = new HttpClient())
            {
                if (String.IsNullOrEmpty(access_token))
                {
                    var Web = new WebWindow.LoginLiveID();
                    access_token = await Web.Login();
                }
                try
                {
                    client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                    var posturl = OneDriveControl.APIbase + url;
                    var req = new HttpRequestMessage(HttpMethod.Post, posturl);
                    req.Headers.Add("Authorization", "Bearer " + access_token);

                    string data = "";
                        

                    var content = new StringContent(data, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeWithQualityHeaderValue("application/json");
                    req.Content = content;
                    using (var response = await client.SendAsync(req, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await CreateUploadSession(ct);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                        errorstr = null;
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
            }
            if (String.IsNullOrEmpty(ret)) return null;
            var serializer = new DataContractJsonSerializer(typeof(UploadSession));
            using (var ms = new MemoryStream(Encoding.UTF8.GetBytes(ret)))
            {
                return (UploadSession)serializer.ReadObject(ms);
            }
        }

        private async Task<string> UploadContent_Fragment(Stream uploadcontent, CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            if (String.IsNullOrEmpty(access_token))
            {
                var Web = new WebWindow.LoginLiveID();
                access_token = await Web.Login(ct);
            }
            var len = uploadcontent.Length;
            using (var client = new HttpClient(new OneDriveControl.RetryHandler(new HttpClientHandler())))
            using (var content = new StreamContent(uploadcontent))
            {
                client.Timeout = TimeSpan.FromDays(1);
                try
                {
                    content.Headers.Add("Content-Range", String.Format("bytes {0}-{1}/{2}", offset, offset + len - 1, TotalByte));

                    string target_uri = uploadurl;
                    target_uri += "?access_token=" + Uri.EscapeUriString(access_token);
                    using (var response = await client.PutAsync(target_uri, content, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await UploadContent_Fragment(uploadcontent, ct);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }

            }
            offset += len;
            return ret;
        }
    }

    public partial class OneDriveControl
    {
        private string access_token;
        public string errorstr { get; private set; }
        public const string APIbase = "https://api.onedrive.com/v1.0";


        public async Task<string> UploadContent_withPath(Stream uploadcontent, string filename, string target_path, CancellationToken ct = default(CancellationToken))
        {
            var uploader = new UploadResume(uploadcontent, filename, "/drive/root:" + target_path + "/" + filename + ":/upload.createSession");
            return await uploader.Upload();
        }

        public async Task<string> UploadContent_withId(Stream uploadcontent, string filename, string target_id, CancellationToken ct = default(CancellationToken))
        {
            var uploader = new UploadResume(uploadcontent, filename, "/drive/items/"+target_id+":/" +filename +":/upload.createSession");
            return await uploader.Upload();
        }



        public async Task<string> GetFolderJson_withPath(string target_path = null, CancellationToken ct = default(CancellationToken))
        {
            if(String.IsNullOrEmpty(target_path))
                return await GetItemJson("/drive/root/children", ct);
            else
                return await GetItemJson("/drive/root:"+target_path+":/children" ,ct);
        }

        public async Task<string> GetFolderJson_withId(string target_id = null, CancellationToken ct = default(CancellationToken))
        {
            if (String.IsNullOrEmpty(target_id))
                return await GetItemJson("/drive/root/children", ct);
            else
                return await GetItemJson("/drive/items/" + target_id + "/children", ct);
        }

        public async Task<string> GetFileJson_withPath(string target_path = null, CancellationToken ct = default(CancellationToken))
        {
            if (String.IsNullOrEmpty(target_path))
                return await GetItemJson("/drive/root", ct);
            else
                return await GetItemJson("/drive/root:" + target_path , ct);
        }

        public async Task<string> GetFileJson_withId(string target_id = null, CancellationToken ct = default(CancellationToken))
        {
            if (String.IsNullOrEmpty(target_id))
                return await GetItemJson("/drive/root", ct);
            else
                return await GetItemJson("/drive/items/" + target_id , ct);
        }

        public async Task<string> GetJson_Search(string searchstr, CancellationToken ct = default(CancellationToken))
        {
            return await GetItemJson("/drive/root/view.search?q=" + Uri.EscapeUriString(searchstr), ct);
        }

        //http://stackoverflow.com/questions/19260060/retrying-httpclient-unsuccessful-requests
        public class RetryHandler : DelegatingHandler
        {
            // Strongly consider limiting the number of retries - "retry forever" is
            // probably not the most user friendly way you could respond to "the
            // network cable got pulled out."
            private const int MaxRetries = 20;

            public RetryHandler(HttpMessageHandler innerHandler)
                : base(innerHandler)
            { }

            protected override async Task<HttpResponseMessage> SendAsync(
                HttpRequestMessage request,
                CancellationToken cancellationToken)
            {
                HttpResponseMessage response = null;
                for (int i = 0; i < MaxRetries; i++)
                {
                    response = await base.SendAsync(request, cancellationToken);
                    if (response.IsSuccessStatusCode || response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                    {
                        return response;
                    }
                }

                return response;
            }
        }

        private async Task<string> GetItemJson(string target_url, CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            if (String.IsNullOrEmpty(access_token))
            {
                var Web = new WebWindow.LoginLiveID();
                access_token = await Web.Login(ct);
            }
            using (var client = new HttpClient(new RetryHandler(new HttpClientHandler())))
            {
                try
                {
                    string target_uri;
                    if (target_url.StartsWith("https://"))
                    {
                        target_uri = target_url;
                    }
                    else
                    {
                        target_uri = APIbase + target_url;
                        target_uri += (target_uri.Contains('?') ? "&" : "?") + "access_token=" + Uri.EscapeUriString(access_token);
                    }
                    using (var response = await client.GetAsync(target_uri, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await GetItemJson(target_url, ct);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                        errorstr = null;
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
            }
            return ret;
        }


        public async Task<string> UploadContent_Simple_withPath(Stream uploadcontent, string filename, string target_path, CancellationToken ct = default(CancellationToken))
        {
            return await UploadContent_Simple(uploadcontent, "/drive/root:" + target_path + "/" + filename + ":/content", ct);
        }

        public async Task<string> UploadContent_Simple_withId(Stream uploadcontent, string filename, string target_id, CancellationToken ct = default(CancellationToken))
        {
            return await UploadContent_Simple(uploadcontent, "/drive/items/" + target_id + "/children/" + filename + "/content", ct);
        }

        private async Task<string> UploadContent_Simple(Stream uploadcontent, string target_url, CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            if (String.IsNullOrEmpty(access_token))
            {
                var Web = new WebWindow.LoginLiveID();
                access_token = await Web.Login(ct);
            }
            using (var client = new HttpClient(new RetryHandler(new HttpClientHandler())))
            using (var content = new StreamContent(uploadcontent))
            {
                client.Timeout = TimeSpan.FromDays(1);
                try
                {
                    string target_uri = APIbase + Uri.EscapeUriString(target_url);
                    target_uri += "?access_token=" + Uri.EscapeUriString(access_token);
                    using (var response = await client.PutAsync(target_uri, content, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await UploadContent_Simple(uploadcontent, target_url, ct);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }

            }
            return ret;
        }

        public async Task<string> CreateFolder_withPath(string target_path = null, string newFoldername = "", CancellationToken ct = default(CancellationToken))
        {
            if (String.IsNullOrEmpty(target_path))
                return await CreateFolder("/drive/root/children", newFoldername, ct);
            else
                return await CreateFolder("/drive/root:" + target_path + ":/children", newFoldername, ct);
        }

        public async Task<string> CreateFolder_withId(string target_id = null, string newFoldername = "", CancellationToken ct = default(CancellationToken))
        {
            if (String.IsNullOrEmpty(target_id))
                return await CreateFolder("/drive/root/children", newFoldername, ct);
            else
                return await CreateFolder("/drive/items/" + target_id + "/children", newFoldername, ct);
        }

        private async Task<string> CreateFolder(string target_url, string newFoldername, CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            using (var client = new HttpClient())
            {
                if (String.IsNullOrEmpty(access_token))
                {
                    var Web = new WebWindow.LoginLiveID();
                    access_token = await Web.Login();
                }
                try
                {
                    client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                    var url = APIbase + target_url;
                    var req = new HttpRequestMessage(HttpMethod.Post, url);
                    req.Headers.Add("Authorization", "Bearer " + access_token);

                    string data =
                        "{ \"name\": \"" + newFoldername + "\", \"folder\": { } }";

                    var content = new StringContent(data, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeWithQualityHeaderValue("application/json");
                    req.Content = content;
                    using (var response = await client.SendAsync(req, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await CreateFolder(target_url, newFoldername);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                        errorstr = null;
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
            }
            return ret;
        }

        public async Task<string> MoveItem_withPath(string target_path, string move_to_path, CancellationToken ct = default(CancellationToken))
        {
            string data =
                "{\"parentReference\": {\"path\": \"/drive/root:" + move_to_path + "\"}}";
            return await MoveItem("/drive/root:" + target_path, data, ct);
        }

        public async Task<string> MoveItem_withId(string target_id, string move_to_id, CancellationToken ct = default(CancellationToken))
        {
            string data =
                "{\"parentReference\": {\"id\": \"" + move_to_id + "\"}}";
            return await MoveItem("/drive/items/" + target_id, data, ct);
        }

        private async Task<string> MoveItem(string target_url, string data, CancellationToken ct = default(CancellationToken))
        {
            string ret = "";
            ct.ThrowIfCancellationRequested();
            using (var client = new HttpClient())
            {
                if (String.IsNullOrEmpty(access_token))
                {
                    var Web = new WebWindow.LoginLiveID();
                    access_token = await Web.Login();
                }
                try
                {
                    client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                    var url = APIbase + target_url;
                    var req = new HttpRequestMessage(new HttpMethod("PATCH"), url);
                    req.Headers.Add("Authorization", "Bearer " + access_token);

                    var content = new StringContent(data, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeWithQualityHeaderValue("application/json");
                    req.Content = content;
                    using (var response = await client.SendAsync(req, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            return await MoveItem(target_url, data);
                        }
                        response.EnsureSuccessStatusCode();
                        ret = await response.Content.ReadAsStringAsync();
                        errorstr = null;
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
            }
            return ret;
        }

        public async Task DeleteItem_withPath(string target_path, CancellationToken ct = default(CancellationToken))
        {
            await DeleteItem("/drive/root:" + target_path, ct);
        }

        public async Task DeleteItem_withId(string target_id, CancellationToken ct = default(CancellationToken))
        {
            await DeleteItem("/drive/items/" + target_id, ct);
        }

        private async Task DeleteItem(string target_url, CancellationToken ct = default(CancellationToken))
        {
            ct.ThrowIfCancellationRequested();
            using (var client = new HttpClient())
            {
                if (String.IsNullOrEmpty(access_token))
                {
                    var Web = new WebWindow.LoginLiveID();
                    access_token = await Web.Login();
                }
                try
                {
                    var url = APIbase + target_url;
                    url += "?access_token=" + Uri.EscapeUriString(access_token);
                    var req = new HttpRequestMessage(new HttpMethod("DELETE"), url);
                    using (var response = await client.SendAsync(req, ct))
                    {
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            errorstr = "Authorized...";
                            await DeleteItem(target_url, ct);
                            return;
                        }
                        errorstr = null;
                        if (response.StatusCode == System.Net.HttpStatusCode.NoContent)
                        {
                            return;
                        }                       
                        response.EnsureSuccessStatusCode();
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
            }
        }
        public const Int64 DownloadBlock = 16 * 1024 * 1024;
        public const int concurrency = 4;

        private async Task<MemoryStream> DownloadItem(ResponseFolder target, int range = 0, CancellationToken ct = default(CancellationToken))
        {
            var ret = new MemoryStream();
            ct.ThrowIfCancellationRequested();
            if (String.IsNullOrEmpty(access_token))
            {
                var Web = new WebWindow.LoginLiveID();
                access_token = await Web.Login(ct);
            }
            if (target.folder != null) return null;
            if (target.size == null) return null;
            Int64 size = (Int64)target.size;
            Int64 endrange = (range + 1) * DownloadBlock - 1;
            Int64 startrange = range * DownloadBlock;
            int retry = 100;
            while (retry-- > 0)
            {
                if (retry < 99)
                    await Task.Run(() =>
                    {
                        ct.WaitHandle.WaitOne(TimeSpan.FromSeconds(5));
                    }, ct);
                try
                {
                    using (var client = new HttpClient(new RetryHandler(new HttpClientHandler())))
                    {
                        //System.Diagnostics.Debug.WriteLine("_range:" + startrange.ToString()+"-"+endrange.ToString());
                        if (startrange >= size) throw new ArgumentOutOfRangeException("over size start range.");
                        if (endrange >= size) endrange = size - 1;
                        //System.Diagnostics.Debug.WriteLine("range:" + startrange.ToString() + "-" + endrange.ToString());

                        client.DefaultRequestHeaders.Range = new RangeHeaderValue(startrange, endrange);
                        var url = APIbase + "/drive/items/" + target.id + "/content";
                        url += "?access_token=" + Uri.EscapeUriString(access_token);
                        var response = await client.GetAsync(url, HttpCompletionOption.ResponseHeadersRead, ct);
                        if (response.StatusCode == System.Net.HttpStatusCode.Unauthorized)
                        {
                            access_token = "";
                            return await DownloadItem(target, range, ct);
                        }
                        response = await client.GetAsync(url, HttpCompletionOption.ResponseContentRead, ct);
                        response.EnsureSuccessStatusCode();
                        await (await response.Content.ReadAsStreamAsync()).CopyToAsync(ret);
                        errorstr = null;
                    }
                }
                catch (HttpRequestException e)
                {
                    errorstr = e.Message;
                    System.Diagnostics.Debug.WriteLine(errorstr);
                    System.Diagnostics.Debug.WriteLine("retry:" + retry.ToString());
                    continue;
                }
                catch (TaskCanceledException e)
                {
                    ct.ThrowIfCancellationRequested();
                    errorstr = e.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                    System.Diagnostics.Debug.WriteLine("retry:" + retry.ToString());
                    continue;
                }
                catch (Exception ex)
                {
                    errorstr = ex.ToString();
                    System.Diagnostics.Debug.WriteLine(errorstr);
                }
                break;
            }
            ret.Position = 0;
            return ret;
        }

        async Task<FolderRequest> CheckNextItem(FolderRequest item, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            if (string.IsNullOrEmpty(item.nextLink)) return item;
            FolderRequest nextitem = await CheckNextItem(ParseItemJson<FolderRequest>(await GetItemJson(item.nextLink, cancellationToken)), cancellationToken);
            item.value.AddRange(nextitem.value);
            return item;
        }

        async Task<SearchRequest> CheckNextSearchItem(SearchRequest item, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            if (string.IsNullOrEmpty(item.nextLink)) return item;
            SearchRequest nextitem = await CheckNextSearchItem(ParseItemJson<SearchRequest>(await GetItemJson(item.nextLink, cancellationToken)), cancellationToken);
            item.value.AddRange(nextitem.value);
            return item;
        }

        public async Task<FolderRequest> GetFolder_id(string target_id = null, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            return await CheckNextItem(ParseItemJson<FolderRequest>(await GetFolderJson_withId(target_id, cancellationToken)), cancellationToken);
        }
        public async Task<FolderRequest> GetFolder_path(string target_path = null, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            return await CheckNextItem(ParseItemJson<FolderRequest>(await GetFolderJson_withPath(target_path, cancellationToken)), cancellationToken);
        }

        public async Task<ResponseFolder> GetFile_id(string target_id = null, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            return ParseItemJson<ResponseFolder>(await GetFileJson_withId(target_id, cancellationToken));
        }
        public async Task<ResponseFolder> GetFile_path(string target_path = null, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            return ParseItemJson<ResponseFolder>(await GetFileJson_withPath(target_path, cancellationToken));
        }

        public async Task<SearchRequest> GetFolder_search(string target_str = "", CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            return await CheckNextSearchItem(ParseItemJson<SearchRequest>(await GetJson_Search(target_str, cancellationToken)), cancellationToken);
        }

        private T ParseItemJson<T>(string response)
        {
            if (String.IsNullOrEmpty(response)) return default(T);
            var serializer = new DataContractJsonSerializer(typeof(T));
            using (var ms = new MemoryStream(Encoding.UTF8.GetBytes(response)))
            {
                return (T)serializer.ReadObject(ms);
            }
        }



    }

    public static class EnumerableExtensions
    {
        public static async Task ForEachAsync<T>(this IEnumerable<T> source, Func<T, Task> action, int concurrency, CancellationToken cancellationToken = default(CancellationToken), bool configureAwait = false)
        {
            if (source == null) throw new ArgumentNullException("source");
            if (action == null) throw new ArgumentNullException("action");
            if (concurrency <= 0) throw new ArgumentOutOfRangeException("concurrencyは1以上の必要があります");

            using (var semaphore = new SemaphoreSlim(initialCount: concurrency, maxCount: concurrency))
            {
                var exceptionCount = 0;
                var tasks = new List<Task>();

                foreach (var item in source)
                {
                    if (exceptionCount > 0) break;
                    cancellationToken.ThrowIfCancellationRequested();

                    await semaphore.WaitAsync(cancellationToken).ConfigureAwait(configureAwait);
                    var task = action(item).ContinueWith(t =>
                    {
                        semaphore.Release();

                        if (t.IsFaulted)
                        {
                            Interlocked.Increment(ref exceptionCount);
                            System.Diagnostics.Debug.WriteLine("ForEachAsync Failed:" + t.Exception.Message);
                            throw t.Exception;
                        }
                    });
                    tasks.Add(task);
                }

                await Task.WhenAll(tasks.ToArray()).ConfigureAwait(configureAwait);
            }
        }
    }
}

