﻿using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.Threading.Tasks;
using System.Net.Http;
using System.Net.Http.Headers;
using System.IO;
using System.Text;
using System.Threading;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using WebWindow;

namespace OneDrive
{
    public partial class OneDriveControl
    {
        private async Task<MemoryStream> DownloadBuf(ResponseFolder target, CancellationToken cancellationToken = default(CancellationToken))
        {
            MemoryStream ret = new MemoryStream();
            cancellationToken.ThrowIfCancellationRequested();
            if (target.size == 0) return ret;
            var splitcount = (Int32)((target.size - 1) / DownloadBlock) + 1;
            await Enumerable.Range(0, splitcount).ForEachAsync(
                async x =>
                {
                    using (var buf = await DownloadItem(target, x, cancellationToken))
                    {
                        lock (ret)
                        {
                            ret.Position = x * DownloadBlock;
                            buf.WriteTo(ret);
                        }
                    }
                }, concurrency: 10, cancellationToken: cancellationToken);
            ret.Position = 0;
            return ret;
        }

        public async Task<bool> TestMultiBlock(ResponseFolder mother, FolderRequest children, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            int num = 0;
            var header = children.value.Find(x => x.name == "0000");
            if (header != null)
            {
                var head = await DownloadPart(header, 0, cancellationToken);
                byte[] buf = new byte[sizeof(Int32)];
                head.Read(buf, 0, sizeof(Int32));
                num = BitConverter.ToInt32(buf, 0);

                int ver = (num & 0x7F000000) >> 24;
                num &= 0xFFFFFF;
            }
            else return false;
            if (num <= 0) return false;
            if ((mother.folder == null) || (mother.folder.childCount == null)) return false;
            return (mother.folder.childCount == num + 1);
        }

        public async Task<bool> TestMultiBlock_path(string target_path, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            FolderRequest children = await GetFolder_path(target_path, cancellationToken);
            ResponseFolder mother = await GetFile_path(target_path, cancellationToken);
            return await TestMultiBlock(mother, children, cancellationToken);
        }

        public async Task<bool> TestMultiBlock_id(string target_id, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            FolderRequest children = await GetFolder_id(target_id, cancellationToken);
            ResponseFolder mother = await GetFile_id(target_id, cancellationToken);
            return await TestMultiBlock(mother, children, cancellationToken);
        }

        public async Task DownloadMultiBlock(string target, string filename, CancellationToken cancellationToken = default(CancellationToken))
        {
            byte[] salt;
            string password = Config.password_prefix;
            string orgFilename;
            byte[] iv;
            var rijndael = new RijndaelManaged();
            rijndael.KeySize = 256;

            // follow path
            FolderRequest children = await GetFolder_path(target, cancellationToken);
            ResponseFolder mother = await GetFile_path(target, cancellationToken);

            int num = 0;
            SHA512Managed hasher = new SHA512Managed();
            byte[] hash = new byte[hasher.HashSize / 8];

            var header = children.value.Find(x => x.name == "0000");
            if (header != null)
            {
                var head = await DownloadPart(header, 0, cancellationToken);
                byte[] buf = new byte[sizeof(Int32)];
                head.Read(buf, 0, sizeof(Int32));
                num = BitConverter.ToInt32(buf, 0);

                int ver = (num & 0x7F000000) >> 24;
                num &= 0xFFFFFF;

                if (ver == 0)
                    rijndael.BlockSize = 256;
                else
                    rijndael.BlockSize = 128;

                iv = new byte[rijndael.BlockSize / 8];
                head.Read(iv, 0, iv.Length);

                salt = new byte[SaltSize];
                head.Read(salt, 0, salt.Length);
                int strlen = 0;
                for (int i = 0; i < 4; i++)
                {
                    byte[] p = new byte[1];
                    head.Read(p, 0, 1);
                    strlen |= ((p[0] & 0x7F) << 7 * i);
                    if ((p[0] & 0x80) == 0) break;
                }
                var strbuf = new byte[strlen];
                head.Read(strbuf, 0, strbuf.Length);
                orgFilename = Encoding.Unicode.GetString(strbuf);

                head.Read(hash, 0, hash.Length);
            }
            else return;
            if (num <= 0) return;
            if ((mother.folder == null) || (mother.folder.childCount == null)) return;
            if (mother.folder.childCount != num + 1) return;

            password += orgFilename;
            var DeriveBytes = new Rfc2898DeriveBytes(password, salt, myIterations);
            rijndael.Key = DeriveBytes.GetBytes(rijndael.KeySize / 8);
            rijndael.IV = iv;
            ICryptoTransform decryptor = rijndael.CreateDecryptor();
            DeriveBytes.Reset();

            if (ChangeFilenameOneDrive(orgFilename) == Path.GetFileName(filename))
            {
                // recover org filename
                filename = Path.GetDirectoryName(filename) + "\\" + orgFilename;
            }

            int concurrency = 10;
            children.value.RemoveAll(x => x.name == "0000");
            using (var destination = new FileStream(filename, FileMode.Create, FileAccess.Write, FileShare.Read, bufferSize: 512 * 1024, useAsync: true))
            {
                IDecryptStream dest = new DecryptStreamWriter(decryptor, hasher, cancellationToken);
                Task DecryptTask = null;
                foreach (var downtask in from aFile in children.value orderby int.Parse(aFile.name) group aFile by int.Parse(aFile.name) / concurrency)
                {
                    // Create a query. 
                    IEnumerable<Task<MemoryStream>> downloadTasksQuery =
                        from aFile in downtask select DownloadBuf(aFile, cancellationToken);


                    // Use ToArray to execute the query and start the download tasks.
                    Task<MemoryStream>[] downloadTasks = downloadTasksQuery.ToArray();

                    // Await the completion of all the running tasks.
                    MemoryStream[] allbuf = await Task.WhenAll(downloadTasks);

                    if (DecryptTask != null) await DecryptTask;

                    DecryptTask = Task.Run(async () =>
                    {
                        foreach (var buf in allbuf)
                        {
                            cancellationToken.ThrowIfCancellationRequested();
                            await dest.DecryptStream(destination, buf);
                        }
                    }, cancellationToken);
                }
                if (DecryptTask != null) await DecryptTask;
                await dest.DecryptStreamFinal(destination);
            }
            if (!hasher.Hash.SequenceEqual(hash))
            {
                using (var dest = File.Create(filename + ".log"))
                using (var sw = new StreamWriter(dest))
                {
                    sw.WriteLine(filename);
                    sw.WriteLine("download hash:");
                    sw.Write(BitConverter.ToString(hasher.Hash).Replace("-", ""));
                    sw.WriteLine("");
                    sw.WriteLine("upload hash:");
                    sw.Write(BitConverter.ToString(hash).Replace("-", ""));
                    sw.WriteLine("");
                }
            }
        }
    
        public async Task Download(string target, string filename, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            var target_folder = await GetFile_path(target, cancellationToken);
            if (target_folder == null) return;

            if (target_folder.folder != null)
            {
                await DownloadMultiBlock(target, filename, cancellationToken);
                return;
            }

            await Download(target_folder, filename, cancellationToken);
        }

        private async Task Download(ResponseFolder target, string filename, CancellationToken cancellationToken = default(CancellationToken))
        {
            cancellationToken.ThrowIfCancellationRequested();
            using (Stream destination = File.Create(filename))
            {
                if (target.size == 0) return;
                var splitcount = (Int32)((target.size - 1) / DownloadBlock) + 1;
                await Enumerable.Range(0, splitcount).ForEachAsync(
                    async x =>
                    {
                        using (var buf = await DownloadItem(target, x, cancellationToken))
                        {
                            if (String.IsNullOrEmpty(errorstr))
                            {
                                destination.Seek(x * DownloadBlock, SeekOrigin.Begin);
                                buf.WriteTo(destination);
                            }
                        }
                    }, concurrency: 10, cancellationToken: cancellationToken);
            }
        }

    }

    interface IDecryptStream
    {
        Task DecryptStream(Stream outStream, Stream buf);
        Task DecryptStreamFinal(Stream outStream);
        void SetCancellationToken(CancellationToken cancellationToken);
    }

    public class DecryptStreamWriter: IDecryptStream
    {
        private byte[] InternalBuffer = null;
        ICryptoTransform decryptor;
        HashAlgorithm hasher;
        CancellationToken ct;

        public DecryptStreamWriter(ICryptoTransform decryptor, HashAlgorithm hasher, CancellationToken cancellationToken = default(CancellationToken))
        {
            this.decryptor = decryptor;
            this.hasher = hasher;
            this.ct = cancellationToken;
        }

        public void SetCancellationToken(CancellationToken cancellationToken)
        {
            this.ct = cancellationToken;
        }

        public async Task DecryptStream(Stream outStream, Stream buf)
        {
            Queue<Task> writeTask = new Queue<Task>();
            byte[] outbuffer = new byte[decryptor.OutputBlockSize];
            buf.Position = 0;
            while (buf.Position < buf.Length)
            {
                ct.ThrowIfCancellationRequested();
                if (InternalBuffer != null)
                {
                    var bytesWrite = decryptor.TransformBlock(InternalBuffer, 0, InternalBuffer.Length, outbuffer, 0);
                    hasher.TransformBlock(outbuffer, 0, bytesWrite, outbuffer, 0);
                    writeTask.Enqueue(outStream.WriteAsync(outbuffer, 0, bytesWrite, ct));
                }
                if (buf.Length - buf.Position < decryptor.InputBlockSize) throw new DataMisalignedException();
                InternalBuffer = new byte[decryptor.InputBlockSize];
                buf.Read(InternalBuffer, 0, InternalBuffer.Length);
                if (writeTask.Count > 128) await writeTask.Dequeue();
            }
            while (writeTask.Count > 0) await writeTask.Dequeue();
        }

        public async Task DecryptStreamFinal(Stream outStream)
        {
            ct.ThrowIfCancellationRequested();
            if (InternalBuffer != null)
            {
                var outbuffer = decryptor.TransformFinalBlock(InternalBuffer, 0, InternalBuffer.Length);
                hasher.TransformFinalBlock(outbuffer, 0, outbuffer.Length);
                await outStream.WriteAsync(outbuffer, 0, outbuffer.Length, ct);
            }
        }
    }

}
