﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.IO;
using System.Threading;


namespace OneDriveView
{
    class ConsoleFunc
    {
        protected static CancellationTokenSource cts = new CancellationTokenSource();

        public async static Task<int> MainFunc(string[] args)
        {
            //http://www.zghthy.com/1322994/codep1/attachconsole-shows-data-on-pipe-but-the-%3E-operator-doesnt-correctly-redirect-on-file
            if (IsRedirected(GetStdHandle(StandardHandle.Output)))
            {
                var initialiseOut = Console.Out;
            }

            bool errorRedirected = IsRedirected(GetStdHandle(StandardHandle.Error));
            if (errorRedirected)
            {
                var initialiseError = Console.Error;
            }

            if (!AttachConsole(-1))
                AllocConsole();

            if (!errorRedirected)
                SetStdHandle(StandardHandle.Error, GetStdHandle(StandardHandle.Output));

            Console.Error.WriteLine("");
            
            Console.CancelKeyPress += new ConsoleCancelEventHandler(CtrlC_Handler);
            switch (args[0])
            {
                case "help":
                    Console.WriteLine("usage");
                    Console.WriteLine("\thelp                                      : show help");
                    Console.WriteLine("\tlist (REMOTE_PATH) [--enc]                : list remote path");
                    Console.WriteLine("\t\t--enc : list only encrypted folder");
                    Console.WriteLine("\tdownload (REMOTE_PATH) (LOCAL_DIR_PATH)   : download item");
                    Console.WriteLine("\tupload (LOCAL_FILE_PATH) (REMOTE_PATH)    : upload item");
                    Console.WriteLine("\tuploadraw (LOCAL_FILE_PATH) (REMOTE_PATH) : upload item without enc");
                    Console.WriteLine("\tsend (REMOTE_PATH)                        : send TS file via UDP");
                    break;
                case "list":
                    Console.Error.WriteLine("list...");
                    return await ListRemote(args);
                case "download":
                    Console.Error.WriteLine("download...");
                    return await Download(args);
                case "upload":
                    Console.Error.WriteLine("upload...");
                    return await UploadEnc(args);
                case "uploadraw":
                    Console.Error.WriteLine("uploadraw...");
                    return await UploadRaw(args);
                case "send":
                    Console.Error.WriteLine("send...");
                    return await SendUDP(args);
                case "search":
                    Console.Error.WriteLine("search...");
                    return await SearchList(args);
            }
            return 0;
        }

        protected static void CtrlC_Handler(object sender, ConsoleCancelEventArgs args)
        {
            Console.Error.WriteLine("");
            Console.Error.WriteLine("Cancel...");
            cts.Cancel();
            args.Cancel = true;
        }

        static async Task<int> ListRemote(string[] args)
        {
            var paramArgs = from p in args where p.StartsWith("--") select p.Substring(2);
            var targetArgs = (from p in args where !p.StartsWith("--") select p).ToArray();
            bool encpathflag = false;
            foreach (var p in paramArgs)
            {
                switch (p)
                {
                    case "enc":
                        Console.Error.WriteLine("(--enc: encrypted check mode)");
                        encpathflag = true;
                        break;
                }
            }

            string remotepath = "";
            OneDrive.ResponseFolder folder = null;
            if (targetArgs.Length > 1)
            {
                remotepath = targetArgs[1];
                remotepath = remotepath.Replace('\\','/');
                if (remotepath.LastIndexOf('/') == remotepath.Length - 1)
                    remotepath = remotepath.Substring(0, remotepath.Length - 1);
                
            }
            else
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var tree = new OneDrive.FormOneDriveTree())
                    {
                        tree.Text = "Select list up item";
                        if (tree.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        folder = tree.selecteditem;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (folder == null) return 0;
            }
            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                OneDrive.FolderRequest res;
                if (folder == null)
                    res = await onedrive.GetFolder_path(remotepath, cts.Token);
                else
                    res = await onedrive.GetFolder_id(folder.id, cts.Token);
                cts.Token.ThrowIfCancellationRequested();
                List<string> errordir = new List<string>();
                foreach (var item in res.value)
                {
                    cts.Token.ThrowIfCancellationRequested();
                    if (encpathflag)
                    {
                        if (item.folder != null)
                        {
                            cts.Token.ThrowIfCancellationRequested();
                            if (await onedrive.TestMultiBlock_id(item.id, cts.Token))
                                Console.WriteLine(item.name);
                            else
                                errordir.Add(item.name);
                        }
                    }
                    else
                    {
                        Console.WriteLine(item.name + ((item.folder != null) ? "/" : ""));
                    }
                }
                if (errordir.Count > 0){
                    Console.WriteLine("");
                    Console.WriteLine("error files");
                    errordir.ForEach(i => Console.WriteLine("(error)\t{0}", i));
                }
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }
        
        static async Task<int> Download(string[] args)
        {
            string remotepath = null;
            string localpath = null;
            if (args.Length > 2)
                localpath = args[2];
            if (args.Length > 1)
            {
                remotepath = args[1];
                remotepath = remotepath.Replace('\\', '/');
                if (remotepath.LastIndexOf('/') == remotepath.Length - 1)
                    remotepath = remotepath.Substring(0, remotepath.Length - 1);

            }

            OneDrive.ResponseFolder folder = null;
            if (String.IsNullOrEmpty(remotepath))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var tree = new OneDrive.FormOneDriveTree())
                    {
                        tree.Text = "Select download item";
                        if (tree.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        folder = tree.selecteditem;
                        remotepath = tree.selectedFullpath;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (folder == null) return 0;
            }

            if (String.IsNullOrEmpty(localpath))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var save = new System.Windows.Forms.SaveFileDialog())
                    {
                        save.FileName = (folder == null)? Path.GetFileName(remotepath) :folder.name;
                        if (save.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        localpath = save.FileName;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (localpath == null) return 0;
            }

            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                if (folder == null)
                    folder = await onedrive.GetFile_path(remotepath, cts.Token);
                cts.Token.ThrowIfCancellationRequested();
                if (Path.GetFileName(localpath) == "")
                {
                    localpath = localpath + folder.name;
                }
                Console.Error.WriteLine("remote:" + remotepath);
                Console.Error.WriteLine("local:" + localpath);
                await onedrive.Download(remotepath, localpath, cts.Token);
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }
        
        static async Task<int> UploadEnc(string[] args)
        {
            string remotepath = null;
            string uploadfile = null;
            if (args.Length > 2)
            {
                remotepath = args[2];
                remotepath = remotepath.Replace('\\', '/');
                if (remotepath.LastIndexOf('/') == remotepath.Length - 1)
                    remotepath = remotepath.Substring(0, remotepath.Length - 1);

            }
            if (args.Length > 1)
                uploadfile = args[1];

            if (String.IsNullOrEmpty(uploadfile))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var open = new System.Windows.Forms.OpenFileDialog())
                    {
                        if (open.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        uploadfile = open.FileName;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (uploadfile == null) return 0;
            }

            OneDrive.ResponseFolder folder = null;
            if (String.IsNullOrEmpty(remotepath))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var tree = new OneDrive.FormOneDriveTree())
                    {
                        tree.Text = "Select upload to";
                        if (tree.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        folder = tree.selecteditem;
                        remotepath = tree.selectedFullpath;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (folder == null) return 0;
            }

            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                if (folder == null)
                    folder = await onedrive.GetFile_path(remotepath, cts.Token);
                Console.Error.WriteLine("local:" + uploadfile);
                Console.Error.WriteLine("remote:" + remotepath);
                await onedrive.UploadFileSeparateAsync(uploadfile, folder.id, cts.Token);
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }
        
        static async Task<int> UploadRaw(string[] args)
        {
            string remotepath = null;
            string uploadfile = null;
            if (args.Length > 2)
            {
                remotepath = args[2];
                remotepath = remotepath.Replace('\\', '/');
                if (remotepath.LastIndexOf('/') == remotepath.Length - 1)
                    remotepath = remotepath.Substring(0, remotepath.Length - 1);

            }
            if (args.Length > 1)
                uploadfile = args[1];

            if (String.IsNullOrEmpty(uploadfile))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var open = new System.Windows.Forms.OpenFileDialog())
                    {
                        if (open.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        uploadfile = open.FileName;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (uploadfile == null) return 0;
            }

            OneDrive.ResponseFolder folder = null;
            if (String.IsNullOrEmpty(remotepath))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var tree = new OneDrive.FormOneDriveTree())
                    {
                        tree.Text = "Select upload to";
                        if (tree.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        folder = tree.selecteditem;
                        remotepath = tree.selectedFullpath;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (folder == null) return 0;
            }

            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                Console.Error.WriteLine("local:" + uploadfile);
                Console.Error.WriteLine("remote:" + remotepath);
                using (var source = File.Open(uploadfile, FileMode.Open))
                {
                    if(folder != null)
                        await onedrive.UploadContent_Simple_withId(source, OneDrive.OneDriveControl.ChangeFilenameOneDrive(Path.GetFileName(uploadfile)), folder.id, cts.Token);
                    else
                        await onedrive.UploadContent_Simple_withPath(source, OneDrive.OneDriveControl.ChangeFilenameOneDrive(Path.GetFileName(uploadfile)), remotepath, cts.Token);
                }
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }
        
        static async Task<int> SendUDP(string[] args)
        {
            string remotepath = null;
            if (args.Length > 1)
            {
                remotepath = args[1];
                remotepath = remotepath.Replace('\\', '/');
                if (remotepath.LastIndexOf('/') == remotepath.Length - 1)
                    remotepath = remotepath.Substring(0, remotepath.Length - 1);

            }
            string startstr = "";
            string durationstr = "";

            OneDrive.ResponseFolder folder = null;
            if (String.IsNullOrEmpty(remotepath))
            {
                Thread t = new Thread(new ThreadStart(() =>
                {
                    using (var tree = new OneDrive.FormOneDriveTree())
                    {
                        tree.Text = "Select download item";
                        if (tree.ShowDialog() != System.Windows.Forms.DialogResult.OK) return;
                        folder = tree.selecteditem;
                        remotepath = tree.selectedFullpath;
                    }
                }));
                t.SetApartmentState(System.Threading.ApartmentState.STA);
                t.Start();
                t.Join();
                if (folder == null) return 0;
            }

            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                Console.Error.WriteLine("remote:" + remotepath);
                double start, duration;
                if (!double.TryParse(startstr, out start)) start = 0;
                if (!double.TryParse(durationstr, out duration)) duration = 0;
                Console.Error.WriteLine("start:" + start.ToString());
                Console.Error.WriteLine("duration:" + duration.ToString());
                await onedrive.SendUDP_byPath(remotepath, startsec: start, durationsec: duration, cancellationToken: cts.Token);
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }

        static async Task<int> SearchList(string[] args)
        {
            var paramArgs = from p in args where p.StartsWith("--") select p.Substring(2);
            var targetArgs = (from p in args where !p.StartsWith("--") select p).ToArray();

            string searchstr = "";
            if (targetArgs.Length > 1)
            {
                searchstr = targetArgs[1];
            }
            else
            {
                return 0;
            }
            var onedrive = new OneDrive.OneDriveControl();
            try
            {
                var res = await onedrive.GetFolder_search(searchstr, cts.Token);
                Console.WriteLine("SearchResult:" + res.count.ToString());
                foreach (var item in res.value)
                {
                    cts.Token.ThrowIfCancellationRequested();
                    Console.WriteLine(item.name + ((item.folder != null) ? "/" : ""));
                }
                return 0;
            }
            catch (OperationCanceledException)
            {
                return -1;
            }
            catch (DirectoryNotFoundException)
            {
                Console.Error.WriteLine("Not Found.");
                return 1;
            }
        }


        [DllImport("kernel32.dll" ,SetLastError = true)]
        private static extern bool AllocConsole();
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern bool AttachConsole(int dwProcessId);
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern IntPtr GetStdHandle(StandardHandle nStdHandle);
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern bool SetStdHandle(StandardHandle nStdHandle, IntPtr handle);
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern FileType GetFileType(IntPtr handle);

        private enum StandardHandle : uint
        {
            Input = unchecked((uint)-10),
            Output = unchecked((uint)-11),
            Error = unchecked((uint)-12)
        }

        private enum FileType : uint
        {
            Unknown = 0x0000,
            Disk = 0x0001,
            Char = 0x0002,
            Pipe = 0x0003
        }

        private static bool IsRedirected(IntPtr handle)
        {
            FileType fileType = GetFileType(handle);

            return (fileType == FileType.Disk) || (fileType == FileType.Pipe);
        }
    }
}
